# 多GPU的简洁实现
:label:`sec_multi_gpu_concise`

每个新模型的并行计算都从零开始实现是无趣的。此外，优化同步工具以获得高性能也是有好处的。下面我们将展示如何使用深度学习框架的高级API来实现这一点。数学和算法与 :numref:`sec_multi_gpu`中的相同。不出所料，你至少需要两个GPU来运行本节的代码。




```python
# 在notebook下面单GPU可以执行，多GPU会提示使用python -m paddle.distributed.launch xx.py格式方式执行。
import paddle
from paddle import nn
import d2l.paddle as d2l
# 导入必要并行训练的依赖包
import paddle.distributed as dist
paddle.set_device("gpu") # 多卡并行需要设置这句
```

## [**简单网络**]

让我们使用一个比 :numref:`sec_multi_gpu`的LeNet更有意义的网络，它依然能够容易地和快速地训练。我们选择的是 :cite:`He.Zhang.Ren.ea.2016`中的ResNet-18。因为输入的图像很小，所以稍微修改了一下。与 :numref:`sec_resnet`的区别在于，我们在开始时使用了更小的卷积核、步长和填充，而且删除了最大汇聚层。



```python
#@save
def resnet18(num_classes, in_channels=1):
    """稍加修改的ResNet-18模型"""
    def resnet_block(in_channels, out_channels, num_residuals,
                     first_block=False):
        blk = []
        for i in range(num_residuals):
            if i == 0 and not first_block:
                blk.append(d2l.Residual(in_channels, out_channels,
                                        use_1x1conv=True, strides=2))
            else:
                blk.append(d2l.Residual(out_channels, out_channels))
        return nn.Sequential(*blk)

    # 该模型使用了更小的卷积核、步长和填充，而且删除了最大汇聚层
    net = nn.Sequential(
        nn.Conv2D(in_channels, 64, kernel_size=3, stride=1, padding=1),
        nn.BatchNorm2D(64),
        nn.ReLU())
    net.add_sublayer("resnet_block1", resnet_block(
        64, 64, 2, first_block=True))
    net.add_sublayer("resnet_block2", resnet_block(64, 128, 2))
    net.add_sublayer("resnet_block3", resnet_block(128, 256, 2))
    net.add_sublayer("resnet_block4", resnet_block(256, 512, 2))
    net.add_sublayer("global_avg_pool", nn.AdaptiveAvgPool2D((1, 1)))
    net.add_sublayer("fc", nn.Sequential(nn.Flatten(),
                                       nn.Linear(512, num_classes)))
    return net
```

## 网络初始化


我们将在训练代码部分初始化网络。
```
    # 初始化并行计算环境
    dist.init_parallel_env()
    
    net = resnet18(10) # resnet10分类
    # 模型设置为数据并行模式
    net = paddle.DataParallel(net) 
    # 模型weight参数初始化，飞桨里默认初始化效果也不错，可以省略
    init_normal = nn.initializer.Normal(mean=0.0, std=0.01)
    for i in net.sublayers():
        if type(i) in [nn.Linear, nn.Conv2D]:        
            init_normal(i.weight)

    # 配置优化器
    trainer = paddle.optimizer.SGD(parameters=net.parameters(), learning_rate=lr)
    # 配置loss
    loss = nn.CrossEntropyLoss()


```

## [**训练**]

如前所述，用于训练的代码需要执行几个基本功能才能实现高效并行：

* 需要在所有设备上初始化网络参数。
* 在数据集上迭代时，要将小批量数据分配到所有设备上。
* 跨设备并行计算损失及其梯度。
* 聚合梯度，并相应地更新参数。

最后，并行地计算精确度和发布网络的最终性能。除了需要拆分和聚合数据外，训练代码与前几章的实现非常相似。

当然，飞桨已经通过并行训练套件帮我们实现了，我们只需要导入和初始化数据并行训练，配置网络模型数据并行，然后写入一个py文件，最后在命令行下使用`python -m paddle.distributed.launch xx.py`直接执行即可。



```python
def train(batch_size, lr):
    devices = d2l.try_all_gpus() # 获得GPU字符串，只是为了最后显示GPU列表，不影响实际训练。

    # 初始化并行计算环境
    dist.init_parallel_env()
    
    net = resnet18(10) # resnet10分类
    # 模型设置为数据并行模式
    net = paddle.DataParallel(net) 
    # 模型weight参数初始化，飞桨里默认初始化效果也不错，可以省略
    init_normal = nn.initializer.Normal(mean=0.0, std=0.01)
    for i in net.sublayers():
        if type(i) in [nn.Linear, nn.Conv2D]:        
            init_normal(i.weight)

    # 配置优化器
    trainer = paddle.optimizer.SGD(parameters=net.parameters(), learning_rate=lr)
    # 配置loss
    loss = nn.CrossEntropyLoss()

    train_iter, test_iter = d2l.load_data_fashion_mnist(batch_size)
    # 设置训练循环次数为10
    timer, num_epochs = d2l.Timer(), 10
    animator = d2l.Animator('epoch', 'test acc', xlim=[1, num_epochs])
    for epoch in range(num_epochs):
        net.train()
        timer.start()
        for batch_id, data in enumerate(train_iter):
            img, label = data
            label.stop_gradient = True

            out = net(img)

            l = loss(input=out, label=label)
            avg_loss = paddle.mean(x=l)
            # 以下两句是为了终端输出精度数据，去掉相关代码不影响实际训练
            acc_top1 = paddle.metric.accuracy(input=out, label=label, k=1)
            acc_top5 = paddle.metric.accuracy(input=out, label=label, k=5)

            if batch_id % 50 == 0: # 每50个batch显示训练信息
                print("[Epoch %d, batch %d] loss: %.5f, acc1: %.5f, acc5: %.5f" % (
                    epoch, batch_id, avg_loss, acc_top1, acc_top5))

            l.backward()
            trainer.step()
            trainer.clear_gradients()
        timer.stop()
        animator.add(epoch + 1, (d2l.evaluate_accuracy_gpu(net, test_iter),))

    print(f'测试精度：{animator.Y[0][-1]:.2f}，{timer.avg():.1f}秒/轮，'
          f'在{str(devices)}')   
```

让我们看看这在实践中是如何运作的。我们先[**在单个GPU上训练网络**]进行预热。

在notebook环境下使用`train(batch_size=256, lr=0.1)`。




```python
# 单卡测试
train(batch_size=256, lr=0.1) 

```

接下来我们[**使用2个GPU进行训练**]。与 :numref:`sec_multi_gpu`中评估的LeNet相比，ResNet-18的模型要复杂得多。这就是显示并行化优势的地方，计算所需时间明显大于同步参数需要的时间。因为并行化开销的相关性较小，因此这种操作提高了模型的可伸缩性。

飞桨的多卡训练，需要在命令行下使用`python -m paddle.distributed.launch xx.py`方式执行。同时飞桨的多卡并行非常智能，会自动识别GPU卡的数量，默认使用全部GPU卡进行训练！

下面我们使用`%%writefile multigpu.py`魔法命令将代码写入文件，然后在notebook中使用`!python -m paddle.distributed.launch multigpu.py`执行即可。

刚才我们知道了，飞桨会自动识别GPU卡的数量，那么神奇的地方就是，在不修改代码和参数，也不修改执行命令的情况下，我们前面写的代码在单卡环境、双卡环境和多卡环境下都能正常运行！




```python
%%writefile multigpu.py
# 飞桨多GPU程序，不管单GPU环境还是多GPU环境，都可以在不修改代码，不修改参数，不修改执行命令的情况下正常执行。
import paddle
from paddle import nn
import d2l.paddle as d2l

# 导入并行训练的依赖包
import paddle.distributed as dist

paddle.set_device("gpu") # 多卡并行需要设置这句

def resnet18(num_classes, in_channels=1):
    """稍加修改的ResNet-18模型"""
    def resnet_block(in_channels, out_channels, num_residuals,
                     first_block=False):
        blk = []
        for i in range(num_residuals):
            if i == 0 and not first_block:
                blk.append(d2l.Residual(in_channels, out_channels,
                                        use_1x1conv=True, strides=2))
            else:
                blk.append(d2l.Residual(out_channels, out_channels))
        return nn.Sequential(*blk)

    # 该模型使用了更小的卷积核、步长和填充，而且删除了最大汇聚层
    net = nn.Sequential(
        nn.Conv2D(in_channels, 64, kernel_size=3, stride=1, padding=1),
        nn.BatchNorm2D(64),
        nn.ReLU())
    net.add_sublayer("resnet_block1", resnet_block(
        64, 64, 2, first_block=True))
    net.add_sublayer("resnet_block2", resnet_block(64, 128, 2))
    net.add_sublayer("resnet_block3", resnet_block(128, 256, 2))
    net.add_sublayer("resnet_block4", resnet_block(256, 512, 2))
    net.add_sublayer("global_avg_pool", nn.AdaptiveAvgPool2D((1,1)))
    net.add_sublayer("fc", nn.Sequential(nn.Flatten(),
                           nn.Linear(512, num_classes)))
    return net

# 训练
def train(batch_size, lr):
    devices = d2l.try_all_gpus() # 获得GPU字符串，只是为了最后显示GPU列表，不影响实际训练。

    # 初始化并行计算环境
    dist.init_parallel_env()
    
    net = resnet18(10) # resnet10分类
    # 模型设置为数据并行模式
    net = paddle.DataParallel(net) 
    # 模型weight参数初始化，飞桨里默认初始化效果也不错，可以省略
    init_normal = nn.initializer.Normal(mean=0.0, std=0.01)
    for i in net.sublayers():
        if type(i) in [nn.Linear, nn.Conv2D]:        
            init_normal(i.weight)

    # 配置优化器
    trainer = paddle.optimizer.SGD(parameters=net.parameters(), learning_rate=lr)
    # 配置loss
    loss = nn.CrossEntropyLoss()

    train_iter, test_iter = d2l.load_data_fashion_mnist(batch_size)
    # 设置训练循环次数为10
    timer, num_epochs = d2l.Timer(), 10
    animator = d2l.Animator('epoch', 'test acc', xlim=[1, num_epochs])
    for epoch in range(num_epochs):
        net.train()
        timer.start()
        for batch_id, data in enumerate(train_iter):
            img, label = data
            label.stop_gradient = True

            out = net(img)

            l = loss(input=out, label=label)
            avg_loss = paddle.mean(x=l)
            # 以下两句是为了终端输出精度数据，去掉相关代码不影响实际训练
            acc_top1 = paddle.metric.accuracy(input=out, label=label, k=1)
            acc_top5 = paddle.metric.accuracy(input=out, label=label, k=5)

            if batch_id % 50 == 0: # 每50个batch显示训练信息
                print("[Epoch %d, batch %d] loss: %.5f, acc1: %.5f, acc5: %.5f" % (
                    epoch, batch_id, avg_loss, acc_top1, acc_top5))

            l.backward()
            trainer.step()
            trainer.clear_gradients()
        timer.stop()
        animator.add(epoch + 1, (d2l.evaluate_accuracy_gpu(net, test_iter),))

    print(f'测试精度：{animator.Y[0][-1]:.2f}，{timer.avg():.1f}秒/轮，'
          f'在{str(devices)}')      

if __name__ == "__main__" :
    # 若为多GPU，学习率需要乘以相应倍数以达到较好效果，比如2GPU lr=0.2 4GPU lr=0.4
    train(batch_size=256, lr=0.2) 
```


```python
# 不管单卡还是多卡环境，下面命令都可以正常执行！
!python -m paddle.distributed.launch multigpu.py
```

## 小结


* 飞桨的多卡并行训练非常简单方便，不用调整原来的网络结构，只要加上三处代码即可。

```
导入并行训练需要的依赖包。

初始化并行环境。

设置数据并行的模型。

```

* 所有的数据并行处理，以及GPU卡的管理，统一由并行模块处理，省心省力。
* 飞桨的多卡并行训练代码不用任何修改，就可以适应单卡、双卡以及多卡的各种环境。


## 练习


1. 本节使用ResNet-18，请尝试不同的迭代周期数、批量大小和学习率，以及使用更多的GPU进行计算。如果使用$16$个GPU（例如，在AWS p2.16xlarge实例上）尝试此操作，会发生什么？
1. 有时候不同的设备提供了不同的计算能力，我们可以同时使用GPU和CPU，那应该如何分配工作？为什么？


[Discussions](https://discuss.d2l.ai/t/2803)

